function [density,zdensity,ndata,gyro,vol,xb,yb,c] = dhist3(data,ctrs,bin2px,imgvol,out,syn)
%This function takes a 2D array and creates a 2D density histogram.
%Synatax:   [density,zdensity,ndata,gyro,vol,xb,yb] = dhist3(data,bins,mod,out);
%Input:     ctrs = the bin centers, will also take bin format e.g.[100 100]
%           data = the array to be binned
%           bin2px = the conversion factor for # of bins to a single pixel
%               Default = 1 bin per pixel.
%           imgvol = the image volume.  Used to calculate clipping.  If
%               empty, then assumed no clipping adjustment.  Default = 0,
%               no clipping.
%           out = remove outlier, 5 SD from median., the value you enter,
%               is the number of SD from mean the outliers are.  default  =
%               [].
%           syn = use synapse number to calculate probability, not the
%               puncta number.  Default = 0 or off, Special case 1, which
%               pops up a dialogue for entering the synapse number.
%Output:    density = a vector of densities equal to bins.
%           zprofile = the z score of each bin
%           rcount = raw count in each bin
%           vol = a vector of volumes equal to bins.
%           xb = the x scale for the 2d hist
%           yb = the y scale for the 2d hist
%           gyro = a structure that contains the raw, volume, density and
%                   plot of the gyro visualization.
%           c = centers of the histogram

if nargin<3||isempty(bin2px)   %defalut
    bin2px = 1;
end
if nargin<4||isempty(imgvol)    %defaults
    imgvol = 0;
end
if nargin<5
    out = [];
end
if nargin<6
    syn = 0;
end

if iscell(ctrs)  %in centers format
    bins(1) = size(ctrs{1},2);
    bins(2) = size(ctrs{2},2);
else    %not in centers format, assume bin format [100 100]
    bins = ctrs;
end

data = unique(data,'rows');     %make sure the data is cleaned of NaNs

[ndata,c] = hist3(data(:,[1 2]),ctrs);    %create the binned data array

xb = linspace(min(data(:,1)),max(data(:,1)),size(ndata,1));       %x scale for the 2d hist
yb = linspace(min(data(:,2)),max(data(:,2)),size(ndata,2));       %y scale for the 2d hist

figure('Name','Raw Count Profile','NumberTitle','off');h = imagesc(xb,yb,ndata');       %plot this graph first.

%now figure out x, y & z to calcuate the volumn cube, rational here is that
%in x, y space the space is the same, only in z is there a volume
%difference.
x = repmat(c{1,1},bins(1),1);      %map the x to the spatial fields
y = repmat(c{1,2}',1,bins(2));     %map the y to the spatial fields
x = vectorize(x);               %vectorize x
y = vectorize(y);               %vectorize y
origin = zeros(size(x,1),2);    %create the origin array, 0,0
z_len = dddist([x y],origin).*bin2px;   %calculate the z length for the array and convert to pixel distance
x_len = abs(c{1,1}(2)-c{1,1}(1));   %calculate the x length for the array, if we are using ctrs, they are already in pixel distances
y_len = abs(c{1,2}(2)-c{1,1}(1));   %calculate the y length for the array
%now convert the length to field arrays
z_len = reshape(z_len,bins(1),bins(2))';
x_len = repmat(x_len,bins(1),bins(2));
y_len = repmat(y_len,bins(1),bins(2));
%now calculate the volume
outer_s = (4/3).*pi.*z_len.^3;  %calculate the outer sphere volume
inner_s = (4/3).*pi.*(z_len-x_len).^3;  %calculate the inner sphere volume 
o_clip = ((pi.*(z_len-y_len/2).^2)/3).*(3*z_len-(z_len-y_len/2))*2;  %calculate the outer clipped region
i_clip = ((pi.*(z_len-x_len-y_len/2).^2)/3).*(3*(z_len-x_len)-(z_len-x_len-y_len/2))*2;     %calcluate the inner clipped region
%the shell sliver volume is the outer sphere volume - clipped region -
%inner sphere volume - clipped region divided by two, because only one
%hemisphere
vol = ((outer_s-o_clip)-(inner_s-i_clip));  %divide by 2 is unnecessary as the volume is already doubled
%if there is an entered image volume normalize for volume
if imgvol
    y_per = normclip(imgvol(1),z_len);  %calculate the percent clipped for y axis
    x_per = normclip(imgvol(2),z_len);  %calculate the percent clipped for x axis
    z_per = normclip(imgvol(3),z_len);  %calculate the percent clipped for z axis
    total_per = y_per+x_per+z_per;  %total amount of clipping for the volume
    %now adjust the vol
    vol = vol-(vol.*total_per);
end

%now calculate the density
density = ndata./vol;       %get the density value.

%outlier
if ~isempty(out)
    out_thresh = vectorize(density);
    out_thresh(out_thresh==0) = [];
    out_thresh = std(out_thresh)*out;
    %density(density>out_thresh) = out_thresh;
    %we want to smooth our outliers now.
    [ox,oy] = find(density>out_thresh);
    for i = 1:size(ox,1)    %step through each outlier and normalize
        density(ox(i),oy(i)) = mean(mean(select_vol(density,ox(i),oy(i))));
    end
end

%now calculate the gyro view: new volume combine the sides of the rotations
%then mirror.
r = floor(bins(2)/2)-1;       %calculate the rows to flip: -1 for indexing purposes
r2 = ceil(bins(2)/2);      %forward indexing
right = fliplr(ndata(:,end-r:end));   %get to right half of the data and flip it
rvol = fliplr(vol(:,end-r:end));    %do the same for the volume
if rem(bins(2),2)       %if the divison is not even pad one row
%     right = horzcat(right,zeros(1,bins(1)));
%     rvol = horzcat(rvol,zeros(1,bins(1)));
    %double it instead
    right = horzcat(right,ndata(:,r2));
    rvol = horzcat(rvol,vol(:,r2));
end
gdata = ndata(:,1:r2)+right;   %gyro data
gvol = vol(:,1:r2)+rvol;        %gyro volume
gdensity = gdata./gvol;     %gyro density
%outlier
if ~isempty(out)
    %process gplot too
    out_thresh = vectorize(gdensity);
    out_thresh(out_thresh==0) = [];
    out_thresh = std(out_thresh)*out;
    %gplot(gplot>out_thresh) = out_thresh;
    %we want to smooth our outliers now.
    [ox,oy] = find(gdensity>out_thresh);
    for i = 1:size(ox,1)    %step through each outlier and normalize
        gdensity(ox(i),oy(i)) = mean(mean(select_vol(gdensity,ox(i),oy(i))));
    end
end
gplot = horzcat(gdensity,fliplr(gdensity(:,1:r+1)));  %mirror the graph for plotting
%gprob = gplot./(sum(sum(gplot)));   %calculate the probability plot
%output the gyro data
gyro.raw = gdata;
gyro.vol = gvol;
gyro.density = gdensity;
gyro.plot = gplot;


figure('Name','Density Profile','NumberTitle','off');h2 = imagesc(xb,yb,density');    %graph the density
%figure('Name','Probability Density Profile','NumberTitle','off');h2 = imagesc(xb,yb,gprob');    %graph the probability
figure('Name','Raw Folded Profile','NumberTitle','off');h3 = imagesc(xb,yb,gdata');    %graph the density
figure('Name','Folded Volume','NumberTitle','off');h4 = imagesc(xb,yb,gvol');    %graph the density
figure('Name','Folded Density Profile','NumberTitle','off');h5 = imagesc(xb,yb,gdensity');    %graph the density
figure('Name','Mirror Density Profile','NumberTitle','off');h6 = imagesc(xb,yb,gplot');    %graph the density
sdensity = csaps({xb,yb},gplot',[],{xb,yb});     %cubic spline smooth the images
figure('Name','Cubic Spline Smoothed Density Profile','NumberTitle','off');h7 = imagesc(xb,yb,sdensity);   %graph
%figure('Name','Volume Profile','NumberTitle','off');h4 = imagesc(xb,yb,vol');       %plot volume

%calculate statics: z score
%first calculate the mean and the standard deviation of the density distribution
m = mean2(gplot);
s = std2(gplot);
%now calculate the z score for each bin
zdensity = (gplot-m)./s;  %here the z score
%now plot it
figure('Name','Z Score Profile','NumberTitle','off');h8 = imagesc(xb,yb,zdensity');    %graph the density

%display probability plot
if syn~=0    %synapse number probability cal turned on
    if syn==1   %special case, pop up user dialogue
        syn = str2double(response_box('title','Enter the synapse number','position','center'));
    end
    pplot = (gdata/syn)./gvol;  %creat the probability array half
else                %total puncta number probability cal
    pplot = (gdata/size(data,1))./gvol;  %creat the probability array half
end
%outlier
if ~isempty(out)
    %process gplot too
    out_thresh = vectorize(pplot);
    out_thresh(out_thresh==0) = [];
    out_thresh = std(out_thresh)*out;
    %gplot(gplot>out_thresh) = out_thresh;
    %we want to smooth our outliers now.
    [ox,oy] = find(pplot>out_thresh);
    for i = 1:size(ox,1)    %step through each outlier and normalize
        pplot(ox(i),oy(i)) = mean(mean(select_vol(pplot,ox(i),oy(i))));
    end
end
pplot = horzcat(pplot,fliplr(pplot(:,1:r+1)));
figure('Name','Probability per unit vol','NumberTitle','off');imagesc(xb,yb,pplot');    %graph the probability
gyro.pplot = pplot;

%--------------------------------------------------------------------------
%bin noramlization clipping calculation, returns a percent clipped value
%based in spheres in x y and z
function [clip_per] = normclip(d,r)
%d = dimension (or number of slices)
%r = radius
clip_per = 0;   %initialize
%calculate sphere first.
base_s = (4/3).*pi.*r.^3;
%cycle through each slice
for i = 1:d
    h1 = r-i;    %the height of the clipping
    h1(h1<0) = 0;     %zero it out no negatives
    fclip = ((pi.*h1.^2)./3).*(3.*r-h1);
    h2 = r-(d-i);    %the height of the clipping
    h2(h2<0) = 0;
    bclip = ((pi.*h2.^2)./3).*(3.*r-h2);
    %now calculate the percent clipped
    clip_tmp = ((fclip+bclip)./base_s)./d;    %the volume contribution of this slice in percent
    clip_per = clip_per+clip_tmp;
end
%--------------------------------------------------------------------------
%a little function to select a volume edge irregardless
function [sel_vol] = select_vol(data,x,y)
try     %defualt
    sel_vol = data(x-1:x+1,y-1:y+1);
catch
    %now try to find a working selection
    try
        sel_vol = data(x:x+1,y-1:y+1);
    catch
        try
            sel_vol = data(x-1:x,y-1:y+1);
        catch
            try
                sel_vol = data(x-1:x+1,y:y+1);
            catch
                sel_vol = data(x-1:x+1,y-1:y);
            end
        end
    end
end